﻿using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.GUI;
using System;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using WPF.PresentationLayer.Resources;
using WPF.PresentationLayer.UC115.Events;

namespace WPF.PresentationLayer.UC115.ViewModels
{
    public class SpecimenTabsViewModel : ViewModelBase
    {
        #region Properties
        private Patient patient;
        public Patient Patient
        {
            get
            {
                return patient;
            }
            set
            {
                patient = value;
                RaisePropertyChanged(() => Patient);
            }
        }

        private PatientTASViewModel patientTASViewModel;
        public PatientTASViewModel PatientTASViewModel
        {
            get
            {
                return patientTASViewModel;
            }
            set
            {
                patientTASViewModel = value;
                RaisePropertyChanged(() => PatientTASViewModel);
            }
        }

        private PatientABORhViewModel patientABORhViewModel;
        public PatientABORhViewModel PatientABORhViewModel
        {
            get
            {
                return patientABORhViewModel;
            }
            set
            {
                patientABORhViewModel = value;
                RaisePropertyChanged(() => PatientABORhViewModel);
            }
        }

        private PatientABSViewModel patientABSViewModel;
        public PatientABSViewModel PatientABSViewModel
        {
            get
            {
                return patientABSViewModel;
            }
            set
            {
                patientABSViewModel = value;
                RaisePropertyChanged(() => PatientABSViewModel);
            }
        }

        private PatientDATViewModel patientDATViewModel;
        public PatientDATViewModel PatientDATViewModel
        {
            get
            {
                return patientDATViewModel;
            }
            set
            {
                patientDATViewModel = value;
                RaisePropertyChanged(() => PatientDATViewModel);
            }
        }

        private PatientXMatchViewModel patientXMatchViewModel;
        public PatientXMatchViewModel PatientXMatchViewModel
        {
            get
            {
                return patientXMatchViewModel;
            }
            set
            {
                patientXMatchViewModel = value;
                RaisePropertyChanged(() => PatientXMatchViewModel);
            }
        }

        private PatientAntigenTypingViewModel patientAntigenTypingViewModel;
        public PatientAntigenTypingViewModel PatientAntigenTypingViewModel
        {
            get
            {
                return patientAntigenTypingViewModel;
            }
            set
            {
                patientAntigenTypingViewModel = value;
                RaisePropertyChanged(() => PatientAntigenTypingViewModel);
            }
        }

        private PatientInfoBarViewModel patientInfoBarViewModel;
        public PatientInfoBarViewModel PatientInfoBarViewModel
        {
            get
            {
                return patientInfoBarViewModel;
            }
            set
            {
                patientInfoBarViewModel = value;
                RaisePropertyChanged(() => PatientInfoBarViewModel);
            }
        }

        private string selectedSpecimenUID;
        public string SelectedSpecimenUID
        {
            get
            {
                return selectedSpecimenUID;
            }
            set
            {
                selectedSpecimenUID = value;
                OnSpecimenSelected(selectedSpecimenUID);
                RaisePropertyChanged(() => SelectedSpecimenUID);
            }
        }

        public Guid PatientSpecimenGuid { get; private set; }

        private DateTime? specimenExpirationDateTime;
        public DateTime? SpecimenExpirationDateTime
        {
            get
            {
                return specimenExpirationDateTime;
            }
            set
            {
                specimenExpirationDateTime = value;
                RaisePropertyChanged(() => SpecimenExpirationDateTime);
            }
        }

        private int selectedTabIndex;
        public int SelectedTabIndex
        {
            get
            {
                return selectedTabIndex;
            }
            set
            {
                selectedTabIndex = value;
                RaisePropertyChanged(() => SelectedTabIndex);
            }
        }

        private TabItem _selectedTab;
        public TabItem SelectedTab
        {
            get
            {
                return _selectedTab;
            }
            set
            {
                _selectedTab = value;

                // The tab view's data does not get set immediately. If we wait
                // until the UI is idle then we can set the help
                System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
                {
                    var selectedViewModel = GetSelectedTabViewModel();
                    if (selectedViewModel != null)
                    {
                        VbecsBaseForm.SetThisHelpFile(VbecsBaseForm, selectedViewModel.HelpFileTopic);
                    }
                }));
            }
        }

        public SpecimenTabViewModel GetSelectedTabViewModel()
        {
            if (SelectedTab != null)
            {
                var view = SelectedTab.Content as FrameworkElement;

                if (view != null && SelectedTab != null)
                {
                    return view.DataContext as SpecimenTabViewModel;
                }
            }

            return null;
        }

        #region ToolTips
        /// <summary>
        /// NameToolTip
        /// </summary>
        public string NameToolTip { get { return StrRes.Tooltips.Patient.NameToolTip().ResString; } }
        /// <summary>
        /// SsnToolTip
        /// </summary>
        public string SsnToolTip { get { return StrRes.Tooltips.Patient.SsnToolTip().ResString; } }
        /// <summary>
        /// AboRhToolTip
        /// </summary>
        public string AboRhToolTip
        {
            get
            {
                var toolTip = StrRes.Tooltips.Patient.AboRhToolTip().ResString;
                if (Patient != null)
                {
                    toolTip += "\n" + Patient.GetPatientHistoricABORhForTooltip();
                }

                return toolTip;
            }
        }
        #endregion
        #endregion

        public SpecimenTabsViewModel()
        {
            LogoHeaderText = "Automated Testing - Pending Specimen Tests";
            FormText = "VBECS - " + LogoHeaderText;
            LockFunction = LockFunctions.UC115AutomatedTestingReview;
        }

        public SpecimenTabsViewModel(Patient patient, Guid patientSpecimenGuid, VbecsBaseForm vbecsBaseForm)
            : this()
        {
            Patient = patient;
            SetVbecsBaseForm(vbecsBaseForm, true);

            if (!VbecsBaseForm.LockUseCase(patient.PatientGuid))        //251417 - locking patient to prevent multiple specimens from being accepted
            {
                vbecsBaseForm.DialogResult = System.Windows.Forms.DialogResult.Abort;
                return;
            }

            Messenger.Default.Register<PendingTestReviewedEvent>(this, OnPendingTestReviewedEvent);
            VbecsBaseForm.OnPreviewClosing += PreviewClosing;
            VbecsBaseForm.FormClosed += VbecsBaseForm_FormClosed;

            // Fetch all pending specimen tests
            var pendingTests = PendingSpecimenTest.GetPendingSpecimenTests(patient, patientSpecimenGuid, LogonUser.LogonUserDivisionCode.Trim(), PendingTestStatus.Pending, null);
            SelectedSpecimenUID = pendingTests.Select(x => x.SpecimenUid).FirstOrDefault();
            PatientSpecimenGuid = pendingTests.Select(x => x.PatientSpecimenGuid).FirstOrDefault();

            PatientInfoBarViewModel = new PatientInfoBarViewModel(Patient, VbecsBaseForm);
            PatientTASViewModel = new PatientTASViewModel(patient,
                                                          SelectedSpecimenUID,
                                                          vbecsBaseForm,
                                                          pendingTests.Select(x => x as PendingSpecimenTasModel)
                                                                      .Where(w => w != null).ToList());
            PatientABORhViewModel = new PatientABORhViewModel(patient,
                                                              SelectedSpecimenUID,
                                                              vbecsBaseForm,
                                                              pendingTests.Where(x => x is PendingSpecimenAboRhModel)
                                                                          .Select(s => new SpecimenItemViewModel(s)).ToList());
            PatientABSViewModel = new PatientABSViewModel(patient,
                                                          SelectedSpecimenUID,
                                                          vbecsBaseForm,
                                                          pendingTests.Where(x => x is PendingSpecimenAbsModel)
                                                                      .Select(s => new SpecimenItemViewModel(s)).ToList());

            PatientAntigenTypingViewModel = new PatientAntigenTypingViewModel(patient,
                                                          SelectedSpecimenUID,
                                                          vbecsBaseForm,
                                                          pendingTests.Where(x => x is PendingSpecimenAntigenTypingModel)
                                                                      .Select(s => new SpecimenItemViewModel(s)).ToList());

            PatientDATViewModel = new PatientDATViewModel(patient,
                                                          SelectedSpecimenUID,
                                                          vbecsBaseForm,
                                                          pendingTests.Where(x => x is PendingSpecimenDatModel)
                                                                      .Select(s => new SpecimenItemViewModel(s)).ToList());

            PatientXMatchViewModel = new PatientXMatchViewModel(patient,
                                                                SelectedSpecimenUID,
                                                                vbecsBaseForm,
                                                                pendingTests.Where(x => x is PendingSpecimenXMatchModel)
                                                                            .Select(s => new SpecimenXMatchItemViewModel(s, VbecsBaseForm)).ToList());

            SelectedSpecimenUID = pendingTests.Select(x => x.SpecimenUid).FirstOrDefault();
            SpecimenExpirationDateTime = pendingTests.Where(x => x.SpecimenExpirationDate.HasValue)
                                                     .Select(s => s.SpecimenExpirationDate)
                                                     .FirstOrDefault();

            AutoSelectTabWithResults();
        }

        /// <summary>
        /// Clean up events after the form closed
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void VbecsBaseForm_FormClosed(object sender, System.Windows.Forms.FormClosedEventArgs e)
        {
            if (VbecsBaseForm != null)
            {
                Messenger.Default.Unregister(this);
                VbecsBaseForm.OnPreviewClosing -= PreviewClosing;
                VbecsBaseForm.FormClosed -= VbecsBaseForm_FormClosed;
            }
        }

        /// <summary>
        /// Check for unsaved changes before the form tries to close
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PreviewClosing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            // BR_115.15, BR_2.18
            if (VbecsBaseForm != null)
            {
                VbecsBaseForm.IsDirty = IsDirty();
            }
        }

        /// <summary>
        /// Check for unsaved changes
        /// </summary>
        /// <returns></returns>
        private bool IsDirty()
        {
            return (PatientTASViewModel != null && PatientTASViewModel.IsDirty()) ||
                   (PatientABORhViewModel != null && PatientABORhViewModel.IsDirty()) ||
                   (PatientABSViewModel != null && PatientABSViewModel.IsDirty()) ||
                   (PatientAntigenTypingViewModel != null && PatientAntigenTypingViewModel.IsDirty()) ||
                   (PatientDATViewModel != null && PatientDATViewModel.IsDirty()) ||
                   (PatientXMatchViewModel != null && PatientXMatchViewModel.IsDirty());
        }

        private void AutoSelectTabWithResults()
        {
            var selectedViewModel = GetSelectedTabViewModel();

            if (selectedViewModel != null &&
                selectedViewModel.PendingSpecimenList.Any())
            {
                // Current tab still has pending tests so no need to change
                return;
            }

            if (PatientTASViewModel != null && 
                PatientTASViewModel.PendingSpecimenList.Any())
            {
                SelectedTabIndex = 0;
            }
            else if (PatientABORhViewModel != null &&
                     PatientABORhViewModel.PendingSpecimenList.Any())
            {
                SelectedTabIndex = 1;
            }
            else if (PatientABSViewModel != null &&
                     PatientABSViewModel.PendingSpecimenList.Any())
            {
                SelectedTabIndex = 2;
            }
            else if (PatientDATViewModel != null &&
                     PatientDATViewModel.PendingSpecimenList.Any())
            {
                SelectedTabIndex = 3;
            }
            else if (PatientXMatchViewModel != null &&
                     PatientXMatchViewModel.PendingSpecimenList.Any())
            {
                SelectedTabIndex = 4;
            }
            else if (PatientAntigenTypingViewModel != null &&
                     PatientAntigenTypingViewModel.PendingSpecimenList.Any())
            {
                SelectedTabIndex = 5;
            }
            else
            {
                SelectedTabIndex = -1;
            }
        }

        private void OnSpecimenSelected(string specimenUID)
        {
            Messenger.Default.Send<string>(specimenUID);
        }

        private void OnPendingTestReviewedEvent(PendingTestReviewedEvent payload)
        {
            if (payload != null &&
                payload.PendingTestList.Any(x => x is IPendingSpecimenTestModel))
            {
                AutoSelectTabWithResults();

                if (SelectedTabIndex < 0)
                {
                    VbecsBaseForm.Close();
                }
            }
        }
    }
}
